<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * This file contains the moodle hooks for the submission moodles plugin
 *
 * @package   assignsubmission_moodles
 * @copyright 2012 NetSpot {@link http://www.netspot.com.au}
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
defined('MOODLE_INTERNAL') || die();

/**
 *
 * Callback method for data validation---- required method for AJAXmoodle based moodle API
 *
 * @param stdClass $options
 * @return bool
 */
function assignsubmission_moodles_moodle_validate(stdClass $options) {
    global $USER, $CFG, $DB;

    if ($options->moodlearea != 'submission_moodles' &&
            $options->moodlearea != 'submission_moodles_upgrade') {
        throw new moodle_exception('invalidmoodlearea');
    }
    if (!$submission = $DB->get_record('assign_submission', array('id'=>$options->itemid))) {
        throw new moodle_exception('invalidmoodleitemid');
    }
    $context = $options->context;

    require_once($CFG->dirroot . '/mod/assign/locallib.php');
    static $assignment = null;
    if (is_null($assignment) || $assignment->get_context() != $context) {
        $assignment = new assign($context, null, null);
    }

    if ($assignment->get_instance()->id != $submission->assignment) {
        throw new moodle_exception('invalidcontext');
    }

    return true;
}

/**
 * Permission control method for submission plugin ---- required method for AJAXmoodle based moodle API
 *
 * @param stdClass $options
 * @return array
 */
function assignsubmission_moodles_moodle_permissions(stdClass $options) {
    global $USER, $CFG, $DB;

    if ($options->moodlearea != 'submission_moodles' &&
            $options->moodlearea != 'submission_moodles_upgrade') {
        throw new moodle_exception('invalidmoodlearea');
    }
    if (!$submission = $DB->get_record('assign_submission', array('id'=>$options->itemid))) {
        throw new moodle_exception('invalidmoodleitemid');
    }
    $context = $options->context;

    require_once($CFG->dirroot . '/mod/assign/locallib.php');
    static $assignment = null;
    if (is_null($assignment) || $assignment->get_context() != $context) {
        $assignment = new assign($context, null, null);
    }

    if ($assignment->get_instance()->id != $submission->assignment) {
        throw new moodle_exception('invalidcontext');
    }

    if ($assignment->get_instance()->teamsubmission &&
        !$assignment->can_view_group_submission($submission->groupid)) {
        return array('post' => false, 'view' => false);
    }

    if (!$assignment->get_instance()->teamsubmission &&
        !$assignment->can_view_submission($submission->userid)) {
        return array('post' => false, 'view' => false);
    }

    return array('post' => true, 'view' => true);
}

/**
 * Callback called by moodle::get_moodles() and moodle::add(). Gives an opportunity to enforce blind-marking.
 *
 * @param array $moodles
 * @param stdClass $options
 * @return array
 * @throws moodle_exception
 */
function assignsubmission_moodles_moodle_display($moodles, $options) {
    global $CFG, $DB, $USER;

    if ($options->moodlearea != 'submission_moodles' &&
        $options->moodlearea != 'submission_moodles_upgrade') {
        throw new moodle_exception('invalidmoodlearea');
    }
    if (!$submission = $DB->get_record('assign_submission', array('id'=>$options->itemid))) {
        throw new moodle_exception('invalidmoodleitemid');
    }
    $context = $options->context;
    $cm = $options->cm;
    $course = $options->courseid;

    require_once($CFG->dirroot . '/mod/assign/locallib.php');
    $assignment = new assign($context, $cm, $course);

    if ($assignment->get_instance()->id != $submission->assignment) {
        throw new moodle_exception('invalidcontext');
    }

    if ($assignment->is_blind_marking() && !empty($moodles)) {
        // Blind marking is being used, may need to map unique anonymous ids to the moodles.
        $usermappings = array();
        $guestuser = guest_user();

        // Check group users first.
        $userinteam = false;
        if ($assignment->get_instance()->teamsubmission && has_capability('mod/assign:submit', $context)) {
            $assignment->set_course(get_course($course));
            $userinteam = $assignment->can_edit_group_submission($submission->groupid);
        }

        foreach ($moodles as $moodle) {

            if (has_capability('mod/assign:viewblinddetails', $context) && $USER->id != $moodle->userid) {
                $anonid = $assignment->get_uniqueid_for_user($moodle->userid);
                // Show participant information and the user's full name to users with the view blind details capability.
                $a = new stdClass();
                $a->participantnumber = $anonid;
                $a->participantfullname = $moodle->fullname;
                $moodle->fullname = get_string('blindmarkingviewfullname', 'assignsubmission_moodles', $a);
            } else if ($USER->id == $moodle->userid || $submission->userid == $USER->id || $userinteam) { // phpcs:ignore
                // Do not anonymize the user details for this moodle.
            } else {
                // Anonymize the moodles.
                if (empty($usermappings[$moodle->userid])) {
                    $anonid = $assignment->get_uniqueid_for_user($moodle->userid);
                    // The blind-marking information for this moodleer has not been generated; do so now.
                    $moodleer = new stdClass();
                    $moodleer->firstname = get_string('blindmarkingname', 'assignsubmission_moodles', $anonid);
                    $moodleer->lastname = '';
                    $moodleer->firstnamephonetic = '';
                    $moodleer->lastnamephonetic = '';
                    $moodleer->middlename = '';
                    $moodleer->alternatename = '';
                    $moodleer->picture = 0;
                    $moodleer->id = $guestuser->id;
                    $moodleer->email = $guestuser->email;
                    $moodleer->imagealt = $guestuser->imagealt;

                    // Temporarily store blind-marking information for use in later moodles if necessary.
                    $usermappings[$moodle->userid] = new stdClass();
                    $usermappings[$moodle->userid]->fullname = fullname($moodleer);
                    $usermappings[$moodle->userid]->avatar = $assignment->get_renderer()->user_picture($moodleer,
                            array('size' => 18, 'link' => false));
                }

                // Set blind-marking information for this moodle.
                $moodle->fullname = $usermappings[$moodle->userid]->fullname;
                $moodle->avatar = $usermappings[$moodle->userid]->avatar;
                $moodle->profileurl = null;
            }
        }
    }

    return $moodles;
}

/**
 * Callback to force the userid for all moodles to be the userid of the submission and NOT the global $USER->id. This
 * is required by the upgrade code. Note the moodle area is used to identify upgrades.
 *
 * @param stdClass $moodle
 * @param stdClass $param
 */
function assignsubmission_moodles_moodle_add(stdClass $moodle, stdClass $param) {

    global $DB;
    if ($moodle->moodlearea == 'submission_moodles_upgrade') {
        $submissionid = $moodle->itemid;
        $submission = $DB->get_record('assign_submission', array('id' => $submissionid));

        $moodle->userid = $submission->userid;
        $moodle->moodlearea = 'submission_moodles';
    }
}

